#!/usr/bin/env python3
"""
pdf_bookmaker.py – Generate a 4x6 inch OTP booklet PDF from a CSV of symbols.

Usage:
    python3 pdf_bookmaker.py <input_csv> <output_pdf> [--layout portrait|landscape]

    <input_csv>   – CSV file where each cell is a raw symbol (card, die, coin).
                    The file is read row‑by‑row; blank cells are ignored.
    <output_pdf>  – Destination PDF file (will be overwritten).
    --layout      – Optional layout:
                    * portrait  – two rows per 8.5×11 sheet (default is landscape).
                    * landscape – two columns per 8.5×11 sheet.

The script produces a PDF where each page is exactly 4x6 inch.
Four such pages are placed on a single 8.5×11 inch sheet (two rows × two columns
or two columns × two rows, depending on the layout).  Tiny cross‑hair crop‑marks
are drawn at the outer corners of every 4x6 inch rectangle to aid manual cutting.

Dependencies:
    * Python3
    * reportlab (`pip install --user reportlab`)

Author:  Pix (adapted for the “Guerrilla Encryption” article)
"""

import argparse
import csv
import math
import os
import sys

from reportlab.lib import colors
from reportlab.lib.pagesizes import LETTER, landscape, portrait
from reportlab.lib.units import inch
from reportlab.pdfgen import canvas

# ----------------------------------------------------------------------
# Configuration constants (adjust if you change the booklet design)
# ----------------------------------------------------------------------

# ----- Header (centered at top) -----
header_text = f"Page {page_number}"
c.setFont(FONT_NAME, FONT_SIZE + 2)          # 18pt header
header_w = c.stringWidth(header_text, FONT_NAME, FONT_SIZE + 2)

# Give the header a modest 0.30in gap from the top edge
top_margin = 0.30 * inch
c.drawString((PAGE_W_IN * inch - header_w) / 2,
             PAGE_H_IN * inch - top_margin,
             header_text)

# ----- Grid of symbols with left‑hand line numbers -----
c.setFont(FONT_NAME, FONT_SIZE)

# Row height – based on the chosen line spacing
line_height = FONT_SIZE * LINE_SPACING

# Start the first row just below the header (extra 0.10in cushion)
start_y = PAGE_H_IN * inch - top_margin - 0.10 * inch

# Width of each column, after we reserve the left margin for the spine + gutter
left_margin = (SPINE_MARGIN_IN + 0.35) * inch   # 0.35 in extra gutter
col_width = (PAGE_W_IN * inch - left_margin) / COLS_PER_PAGE

for row in range(ROWS_PER_PAGE):
    y = start_y - row * line_height

    # Left‑hand line label (e.g. “[1]”)
    line_label = f"[{row + 1}]"
    c.drawString(0.02 * inch, y, line_label)

    # Draw the ten symbols for this row, shifted right of the label
    for col in range(COLS_PER_PAGE):
        x = left_margin + col * col_width + 0.04 * inch   # tiny gap from the label
        idx = row * COLS_PER_PAGE + col
        token = symbols[idx]
        c.drawString(x, y, token)

# ----------------------------------------------------------------------
def parse_args():
    parser = argparse.ArgumentParser(description="Generate a 3×5 in OTP booklet PDF.")
    parser.add_argument("csv_file", help="Input CSV file with raw symbols")
    parser.add_argument("pdf_file", help="Output PDF file")
    parser.add_argument(
        "--layout",
        choices=["portrait", "landscape"],
        default="landscape",
        help="Arrange 4 pages per sheet: portrait = 2 rows, landscape = 2 columns (default)",
    )
    return parser.parse_args()


# ----------------------------------------------------------------------
def read_symbols(csv_path):
    """
    Read the CSV and flatten it into a list of symbols (preserving order).
    Empty cells are ignored.
    """
    symbols = []
    with open(csv_path, newline="", encoding="utf-8") as f:
        reader = csv.reader(f)
        for row in reader:
            for cell in row:
                cell = cell.strip()
                if cell:  # ignore empty cells
                    symbols.append(cell)
    return symbols


# ----------------------------------------------------------------------
def chunk_symbols(symbols, per_page):
    """Yield successive chunks of `per_page` symbols."""
    for i in range(0, len(symbols), per_page):
        yield symbols[i : i + per_page]


# ----------------------------------------------------------------------
def draw_crop_marks(c, x0, y0, w, h):
    """
    Draw four tiny cross‑hair crop marks at the corners of a rectangle.
    (x0, y0) is the lower‑left corner of the rectangle.
    """
    cm = CROP_MARK_LEN_PT
    # lower‑left
    c.setStrokeColor(CROP_COLOR)
    c.setLineWidth(0.5)
    c.line(x0 - cm, y0, x0 + cm, y0)          # horizontal
    c.line(x0, y0 - cm, x0, y0 + cm)          # vertical
    # lower‑right
    c.line(x0 + w - cm, y0, x0 + w + cm, y0)
    c.line(x0 + w, y0 - cm, x0 + w, y0 + cm)
    # upper‑left
    c.line(x0 - cm, y0 + h, x0 + cm, y0 + h)
    c.line(x0, y0 + h - cm, x0, y0 + h + cm)
    # upper‑right
    c.line(x0 + w - cm, y0 + h, x0 + w + cm, y0 + h)
    c.line(x0 + w, y0 + h - cm, x0 + w, y0 + h + cm)


# ----------------------------------------------------------------------
def render_page(c, symbols, page_number):
    """
    Render a single 4x6 inch OTP page onto the current canvas origin.
    `symbols` must contain exactly COLS_PER_PAGE * ROWS_PER_PAGE entries.
    """
    assert len(symbols) == COLS_PER_PAGE * ROWS_PER_PAGE

    # ----- Header (centered at top) -----
    header_text = f"Page {page_number}"
    c.setFont(FONT_NAME, FONT_SIZE + 2)
    header_w = c.stringWidth(header_text, FONT_NAME, FONT_SIZE + 2)
    c.drawString((PAGE_W_IN * inch - header_w) / 2,
                 PAGE_H_IN * inch - 0.35 * inch,
                 header_text)

    # ----- Grid of symbols with left‑hand line numbers -----
    c.setFont(FONT_NAME, FONT_SIZE)
    line_height = FONT_SIZE * LINE_SPACING
    start_y = PAGE_H_IN * inch - 0.6 * inch      # space for header
    col_width = (PAGE_W_IN * inch) / COLS_PER_PAGE

    for row in range(ROWS_PER_PAGE):
        y = start_y - row * line_height

        # left‑hand line label, e.g. "[Line 1]"
        line_label = f"[Line {row + 1}]"
        c.drawString(0.02 * inch, y, line_label)

        # Define a left‑margin that includes the spine space plus extra gutter
        left_margin = (SPINE_MARGIN_IN + 0.45) * inch   # 0.45 inch gives a comfortable gap

        # draw the 10 symbols for this row, shifted right so they don’t overlap the label
        for col in range(COLS_PER_PAGE):
            x = left_margin + col * col_width + 0.08 * inch
            idx = row * COLS_PER_PAGE + col
            token = symbols[idx]
            c.drawString(x, y, token)

    # ----- Page number in lower‑right corner (inside the page) -----
    header_text = f"Page {page_number}"
    c.setFont(FONT_NAME, FONT_SIZE + 2)
    header_w = c.stringWidth(header_text, FONT_NAME, FONT_SIZE + 2)
    c.drawString((PAGE_W_IN * inch - header_w) / 2,
                 PAGE_H_IN * inch - 0.30 * inch,
                 header_text)

    # ----- Crop marks around the page -----
    draw_crop_marks(c, 0, 0, PAGE_W_IN * inch, PAGE_H_IN * inch)

    # ----- Footer (bottom center) -----
    footer_text = "TOP SECRET"
    c.setFont(FONT_NAME, FONT_SIZE)
    footer_w = c.stringWidth(footer_text, FONT_NAME, FONT_SIZE)
    c.drawString((PAGE_W_IN * inch - footer_w) / 2,
             0.20 * inch,          # 0.20 in above the bottom edge
             footer_text)


# ----------------------------------------------------------------------
def main():
    args = parse_args()

    # ------------------------------------------------------------------
    # Load symbols from CSV
    # ------------------------------------------------------------------
    symbols = read_symbols(args.csv_file)
    if not symbols:
        sys.exit("ERROR: No symbols found in the CSV file.")

    # ------------------------------------------------------------------
    # Determine how many symbols fit on a single OTP page
    # ------------------------------------------------------------------
    per_page = COLS_PER_PAGE * ROWS_PER_PAGE
    total_pages = math.ceil(len(symbols) / per_page)

    # Pad the symbol list with dummy entries (empty strings) so the last page is full.
    padded_symbols = symbols + [""] * (total_pages * per_page - len(symbols))

    # ------------------------------------------------------------------
    # Prepare PDF canvas
    # ------------------------------------------------------------------
    # We will place 4 OTP pages on each LETTER sheet.
    # Depending on layout, we either split the sheet into 2 columns × 2 rows
    # (landscape) or 2 rows × 2 columns (portrait).
    if args.layout == "landscape":
        page_size = landscape(LETTER)
        cols_on_sheet = 1
        rows_on_sheet = 1
    else:
        page_size = portrait(LETTER)
        cols_on_sheet = 1
        rows_on_sheet = 1

    c = canvas.Canvas(args.pdf_file, pagesize=page_size)

    # ------------------------------------------------------------------
    # Iterate over OTP pages and place them on the sheet
    # ------------------------------------------------------------------
    sheet_page_counter = 0
    page_counter = 1  # human‑readable page numbers start at 1

    for page_chunk in chunk_symbols(padded_symbols, per_page):
        # Determine position on the current sheet
        col_idx = (sheet_page_counter % cols_on_sheet)
        row_idx = (sheet_page_counter // cols_on_sheet) % rows_on_sheet

        # Translate origin to the lower‑left corner of the target 3×5 in rectangle
        x_offset = col_idx * PAGE_W_IN * inch
        y_offset = (rows_on_sheet - 1 - row_idx) * PAGE_H_IN * inch  # origin is bottom‑left
        c.saveState()
        c.translate(x_offset, y_offset)

        # Render the OTP page at this origin
        render_page(c, page_chunk, page_counter)

        c.restoreState()

        sheet_page_counter += 1
        page_counter += 1

        # After we have placed 4 pages (2×2) we start a new sheet.
        if sheet_page_counter == cols_on_sheet * rows_on_sheet:
            c.showPage()
            sheet_page_counter = 0

    # If the last sheet wasn't completely full, still finalize it.
    if sheet_page_counter != 0:
        c.showPage()

    c.save()
    print(f"[INFO] Generated {total_pages} OTP pages → {args.pdf_file}")


if __name__ == "__main__":
    main()